#ifndef LLF_RSA_BUILD_H
#define LLF_RSA_BUILD_H

/*
* All the includes that are needed for code using this module to
* Compile correctly should be #included here.
*/

#include "CE2_public.h"

#ifdef __cplusplus
extern "C"
{
#endif

/*
* Creation date : Mon Mar 05 15:34:05 2007
* Last modified : %modify_time%
*/
/** @file
* \brief This file contains declaration of low level utility
* functions for working with RSA cryptography,
* which worked with LibTomCrypt.
* 
* \version LLF_RSA_BUILD.h#1:csrc:1
* \author Pavel Sasunkevich
* \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
* All Rights reserved
*/

/************************ Defines *****************************/

/************************ Enums *******************************/
/************************ Typedefs ****************************/
/************************ Structs *****************************/
/************************ Public Variables ********************/
/************************ Public Functions ********************/

/**
****************************************************************
* Function Name: 
*  LLF_RSA_Build_PubKey
*
* Inputs:
*  @param UserPubKey_ptr [in/out] - A pointer to the public key structure;
*  @param Exponent_ptr [in] - A pointer to the exponent stream of bytes;
*  @param ExponentSize [in] - The size of the exponent, in bytes;
*  @param Modulus_ptr  [in] - A pointer to the modulus stream of bytes;
*  @param ModulusSize  [in] - The size of the modulus, in bytes.
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code.
*
* \brief \b
* Description:
*  LLF_RSA_Build_PubKey populates a CE2RSAPubKey_t structure with
*  the provided modulus and exponent.
*
* \b
* Algorithm:
*  -# Initialize rsa key values (e and N);
*  -# Convert public exponent (e) and modulus exponent (N)
*     from entire buffer to big number mp_int;
*  -# Extrude public exponent (e) and modulus exponent (N)
*	  from big numbers to output public key buffer.
***************************************************************/
CE2Error_t LLF_RSA_Build_PubKey(
                        CE2_RSAUserPubKey_t *UserPubKey_ptr, /* out */
                        DxUint8_t *Exponent_ptr,	/* in */
                        DxUint16_t ExponentSize,	/* in */
                        DxUint8_t *Modulus_ptr,		/* in */
                        DxUint16_t ModulusSize );	/* in */

/**
****************************************************************
* Function Name: 
*  LLF_RSA_Build_PrivKey
*
* Inputs:
*  @param UserPrivKey_ptr [out] - A pointer to the public key structure.
*  @param PrivExponent_ptr [in] - A pointer to the private exponent stream of bytes;
*  @param PrivExponentSize [in] - The size of the private exponent, in bytes;
*  @param PubExponent_ptr  [in]- A pointer to the public exponent stream of bytes;
*  @param PubExponentSize  [in]- The size of the public exponent, in bytes;
*  @param Modulus_ptr      [in]- A pointer to the modulus stream of bytes.
*  @param ModulusSize      [in]- The size of the modulus, in bytes. 
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code.
*
* \brief \b
* Description:
*  LLF_RSA_Build_PrivKey populates a CE2RSAPrivKey_t structure with
* the provided modulus and exponent, marking the key as a "non-CRT" key
*
* \b
* Algorithm:
*  -# Initialize rsa key values (e and N);
*  -# Convert public exponent (e), private exponent (d),
*     modulus exponent (N) from entire buffer to big number mp_int;
*  -# Extrude public exponent (e), private exponent (d),
*     modulus exponent (N) from big numbers to output public key buffer.
***************************************************************/
CE2Error_t LLF_RSA_Build_PrivKey(
                        CE2_RSAUserPrivKey_t   *UserPrivKey_ptr,	/* out */
	                    DxUint8_t              *PrivExponent_ptr,	/* in */
                        DxUint16_t              PrivExponentSize,	/* in */
                        DxUint8_t              *PubExponent_ptr,	/* in */
                        DxUint16_t              PubExponentSize,	/* in */
                        DxUint8_t              *Modulus_ptr,		/* in */
                        DxUint16_t              ModulusSize );		/* in */

/**
****************************************************************
* Function Name: 
*  LLF_RSA_Build_PrivKeyCRT
*
* Inputs:
*  @param UserPrivKey_ptr [out] - A pointer to the public key structure.
*  @param P_ptr  [in] - A pointer to the first factor stream of bytes;
*  @param PSize  [in] - The size of the first factor, in bytes;
*  @param Q_ptr  [in] - A pointer to the second factor stream of bytes;
*  @param QSize  [in] - The size of the second factor, in bytes;
*  @param dP_ptr [in] - A pointer to the first factor's CRT exponent stream of bytes;
*  @param dPSize [in] - The size of the first factor's CRT exponent, in bytes;
*  @param dQ_ptr [in] - A pointer to the second factor's CRT exponent stream of bytes;
*  @param dQSize [in] - The size of the second factor's CRT exponent, in bytes;
*  @param qInv_ptr [in] - A pointer to the first CRT coefficient stream of bytes;
*  @param qInvSize [in] - The size of the first CRT coefficient, in bytes.
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code.
*
* \brief \b
* Description:
*  LLF_RSA_Build_PrivKeyCRT populates a CE2RSAPrivKey_t structure with
*  the provided parameters, marking the key as a "CRT" key.
*
* \b
* Algorithm:
*  -# Initialize rsa key values:
*     p factor of N (p), q factor of N (q), d mod (p - 1) CRT param (dP),
*     d mod (q - 1) CRT param (dQ) and 1/q mod p CRT param (qP);
*  -# Convert previous inicialized rsa key values from entire buffers
*     to big number mp_int;
*  -# Extrude values of p, q, dP, dQ and qP from big numbers to
*     output public key buffer.
***************************************************************/
CE2Error_t LLF_RSA_Build_PrivKeyCRT(
                        CE2_RSAUserPrivKey_t *UserPrivKey_ptr,	/* out */
                        DxUint8_t *P_ptr, 	/* in */
                        DxUint16_t PSize,	/* in */
                        DxUint8_t *Q_ptr,	/* in */
                        DxUint16_t QSize,	/* in */
                        DxUint8_t *dP_ptr,	/* in */
                        DxUint16_t dPSize,	/* in */
                        DxUint8_t *dQ_ptr,	/* in */
                        DxUint16_t dQSize,	/* in */
                        DxUint8_t *qInv_ptr,	/* in */
                        DxUint16_t qInvSize );	/* in */

/**
****************************************************************
* Function Name: 
*  LLF_RSA_Get_PubKey
*
* Inputs:
*  @param UserPubKey_ptr [in] - A pointer to the public key structure. 
*  @param Exponent_ptr   [out] - A pointer to the exponent stream of bytes
*  @param ExponentSize_ptr [in/out] - the size of the exponent buffer in bytes,
*         it is updated to the actual size of the exponent, in bytes. 
*  @param Modulus_ptr    [in] - A pointer to the modulus stream of bytes.
*  @param ModulusSize_ptr [in/out] - the size of the modulus buffer in bytes,
*         it is updated to the actual size of the modulus, in bytes.
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code.
*
* \brief \b
* Description:
*  LLF_RSA_Get_PubKey gets the e,n public key from the database.
*
* \b
* Algorithm:
*  -# Import RSA key from input key buffer to e and N values;
*  -# Extrude public exponent (e) and modulus exponent (N) and
*     calculate their sizes;
***************************************************************/
CE2Error_t LLF_RSA_Get_PubKey(
                        CE2_RSAUserPubKey_t *UserPubKey_ptr, /* in */
                        DxUint8_t  *Exponent_ptr,		/* out */
                        DxUint16_t *ExponentSize_ptr,	/* in, out */
                        DxUint8_t  *Modulus_ptr,		/* out */
                        DxUint16_t *ModulusSize_ptr );	/* in, out */

/**
****************************************************************
* Function Name: 
*  LLF_RSA_Get_ModSizeFromPubKey
*
* Inputs:
*  @param UserPubKey_ptr [in] - A pointer to the public key structure,
*         as returned by CE2_RSA_Build_PubKey.
*  @param ModulusSize_ptr [out] -  The actual size of the modulus, in bytes;
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code.
*
* \brief \b
* Description:
*   The function gets modulus size from public key.
*
* \b
* Algorithm:
*  -# Get from input public key size of modulus exponent in bits 
*     and convert it into bytes.
***************************************************************/
CE2Error_t LLF_RSA_Get_ModSizeFromPubKey(
                        CE2_RSAUserPubKey_t *UserPubKey_ptr,/* in */
                        DxUint16_t *ModulusSize_ptr );		/* out */

/**
****************************************************************
* Function Name: 
*  LLF_RSA_Get_PrivKey
*
* Inputs:
*  @param UserPrivKey_ptr [in] - A pointer to the private key structure;
*  @param PrivExponent_ptr [out] - A pointer to the exponent stream of bytes;
*  @param PrivExponentSize_ptr [in/out] - the size of the exponent buffer in bytes,
*         it is updated to the actual size of the exponent, in bytes;
*  @param PubExponent_ptr [out] - a pointer to the public exponent stream of bytes;
*  @param PubExponentSize_ptr [in/out] - the size of the exponent buffer in bytes,
*         it is updated to the actual size of the exponent, in bytes;
*  @param Modulus_ptr     [out] - A pointer to the modulus stream of bytes.
*  @param ModulusSize_ptr [in/out] - the size of the modolous buffer in bytes,
*         it is updated to the actual size of the modolous, in bytes.
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code.
*
* \brief \b
* Description:
*  LLF_RSA_Get_PrivKey gets the D, n private key from the database.
*
* \b
* Algorithm:
*  -# Initialize rsa key values: public exponent (e),
*     private exponent (d), modulus exponent (N) using input values;
*  -# Extrude public exponent (e), private exponent (d) and
*     modulus exponent (N) from variables and calculate their sizes.
***************************************************************/
CE2Error_t LLF_RSA_Get_PrivKey(
                        CE2_RSAUserPrivKey_t *UserPrivKey_ptr,		/* in */
                        DxUint8_t            *PrivExponent_ptr,		/* out */
                        DxUint16_t           *PrivExponentSize_ptr,	/* in, out */
                        DxUint8_t            *PubExponent_ptr,		/* out */
                        DxUint16_t           *PubExponentSize_ptr,	/* in, out */
                        DxUint8_t            *Modulus_ptr,			/* out */
                        DxUint16_t           *ModulusSize_ptr );	/* in, out */

/**
****************************************************************
* Function Name: 
*  LLF_RSA_Get_PrivKeyCRT
*
* Inputs:
*  @param UserPrivKey_ptr [in] - a pointer to the publick key structure;
*  @param P_ptr [out] - a pointer to the first factor stream of bytes;
*  @param PSize_ptr [in/out] - the size of the first factor buffer in bytes, it is
*         updated to the actual size of the first factor, in bytes;
*  @param Q_ptr [out] - a pointer to the second factor stream of bytes;
*  @param QSize_ptr [in/out] - the size of the second factor buffer in bytes , it is
*         updated to the actual size of the second factor, in bytes;
*  @param dP_ptr [out] - a pointer to the first factors CRT exponent stream of bytes;
*  @param dPSize_ptr [in/out] - the size of the first factor exponent buffer in bytes, it is
*         updated to the actual size of the first factor exponent, in bytes;
*  @param dQ_ptr [out] - a pointer to the second factors CRT exponent stream of bytes;
*  @param dQSize_ptr [in/out] - the size of the second factors CRT exponent buffer in bytes, it is
*         updated to the actual size of the second factors CRT exponent, in bytes;
*  @param qInv_ptr [out] - a pointer to the first CRT coefficient stream of bytes;
*  @param qInvSize_ptr [in/out] - the size of the first CRT coefficient buffer in bytes, it is
*         updated to the actual size of the first CRT coefficient, in bytes.
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code.
*
* \brief \b
* Description:
*  LLF_RSA_Get_PrivKey gets the p,q,dP,dQ,qP parts of private key
*  from the database.
*
* \b
* Algorithm:
*  -# Initialize rsa key values: p, q, dP, dQ and qP;
*  -# Initialize rsa key using data from entire public key;
*  -# Extrude rsa key variables from p, q, dP, dQ and qP into
*     the output variables.
***************************************************************/
CE2Error_t LLF_RSA_Get_PrivKeyCRT(
                        CE2_RSAUserPrivKey_t *UserPrivKey_ptr,	/* in */
                        DxUint8_t  *P_ptr,		/* out */
                        DxUint16_t *PSize_ptr,	/* in, out */
                        DxUint8_t  *Q_ptr,		/* out */
                        DxUint16_t *QSize_ptr,	/* in */
                        DxUint8_t  *dP_ptr, 	/* out */
                        DxUint16_t *dPSize_ptr,	/* in, out */
                        DxUint8_t  *dQ_ptr,		/* out */
                        DxUint16_t *dQSize_ptr,	/* in, out */
                        DxUint8_t  *qInv_ptr,	/* out */
                        DxUint16_t *qInvSize_ptr);/* in, out */

#ifdef __cplusplus
}
#endif

#endif /* LLF_RSA_BUILD_H */
